﻿using LibGit2Sharp;
using LibGit2Sharp.Handlers;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using Wordbee.Beebox.Extensibility;

namespace Wordbee.Beebox.Extensions.GitPackage
{
	/// <summary>
	/// Class that handles pull/push operations Beebox to/from GIT repo
	/// </summary>
	public class GitProxy
	{

		/// <summary>
		/// The config of connector in project
		/// </summary>
		public GitSettings Settings { get; }


		/// <summary>
		/// Branch name. If null then uses Settings.BranchName
		/// </summary>
		public string Branch { get; }


		/// <summary>
		/// Beebox in-directory
		/// </summary>
		public string InDirectory { get; }


		/// <summary>
		/// Beebox in-directory
		/// </summary>
		public string LogDirectory { get; }


		/// <summary>
		/// Beebox in-directory
		/// </summary>
		public string GitDirectory { get; }


		/// <summary>
		/// Git credentials
		/// </summary>
		public CredentialsHandler CredentialsHandler { get; }


		/// <summary>
		/// Signature with sender name and email. Logged inside target git, for example with pushes.
		/// </summary>
		public Signature Signature { get; }


		/// <summary>
		/// Git commit message when pushing content
		/// </summary>
		public string GitCommitMessage { get; } = "Beebox translations";



		/// <summary>
		/// Any log items
		/// </summary>
		public List<LogItem> LogItems { get; } = new List<LogItem>();



		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="settings"></param>
		/// <param name="inDirectory"></param>
		/// <param name="logDirectory"></param>
		public GitProxy(GitSettings settings, string inDirectory, string logDirectory, string branch = null)
		{
			Settings = settings;
			Branch = branch ?? settings.BranchName;
			InDirectory = inDirectory;
			LogDirectory = logDirectory;
			GitDirectory = Path.Combine(InDirectory, ".git");
			CredentialsHandler = (_url, _user, _type) => new UsernamePasswordCredentials() { Username = settings.Username, Password = settings.Password };
			Signature = new Signature(settings.SignatureName, settings.SignatureEmail, DateTimeOffset.Now);
		}



		/// <summary>
		/// Get if the local git directory already exists
		/// </summary>
		public bool GitDirectoryExists() => Directory.Exists(GitDirectory);



		/// <summary>
		/// Sync git repository to beebox: pull updates
		/// Exceptions all handled!
		/// </summary>
		public void SyncGitToBeebox()
		{
			bool gitExists = GitDirectoryExists();

			if (gitExists)
			{
				PullDirectory(GitDirectory);
			}
			else
			{
				CloneToBeebox();
			}
		}



		/// <summary>
		/// Initial clone of .git directory to beebox folder
		/// </summary>
		public void CloneToBeebox()
		{
			// We inform that we want to initial clone
			LogEvent(true, "Initial clone", $"Starting initial clone from branch '{Branch}' to Beebox project");

			// Do it
			CloneDirectory(GitDirectory);

			// Inform that it was done successfully
			LogEvent(true, "Initial clone", $"Initial clone of git repository, target branch '{Branch}'");
		}




		/// <summary>
		/// Commit all changes done of the files.
		/// Exceptions all handled!
		/// </summary>
		/// <param name="directoryOut"></param>
		/// <param name="config"></param>
		public void SyncBeeboxToGit()
		{
			try
			{
				using (var repo = new Repository(GitDirectory))
				{
					LibGit2Sharp.Commands.Stage(repo, "*");
					try
					{
						repo.Commit(GitCommitMessage, Signature, Signature, new CommitOptions()
						{
							AllowEmptyCommit = false	// That will raise exception if empty commit, see catch below
						});

						repo.Network.Push(repo.Head, new PushOptions()
						{
							CredentialsProvider = CredentialsHandler
						});

						LogEvent(true, "Push translations", $"New translations pushed to git repository, target branch '{Branch}'");
					}
					catch (EmptyCommitException)
					{
						// Nothing to commit
					}
				}
			}
			catch (Exception e)
			{
				LogEvent(false, "Push translations", "Failed to push translation to git repository, target branch '{Branch}'", e);
			}
		}



		/// <summary>
		/// Get all branches from the remote directory.
		/// Note: This will do a fetch to get latest remote branches.
		/// </summary>
		/// <param name="proxy"></param>
		/// <returns></returns>
		public List<string> GetRemoteBranches()
		{
			try
			{
				// Create initial clone if not exists
				if (!GitDirectoryExists())
				{
					CloneToBeebox();
				}

				using (var repo = new Repository(GitDirectory))
				{
					// Do a fetch origin to local
					LibGit2Sharp.Commands.Fetch(repo, GitSettings.OriginName, new string[0], new FetchOptions { CredentialsProvider = CredentialsHandler }, null);

					// Get branches from local
					return repo.Branches.Where(b => b.IsRemote).Select(b => cleanBranchName(b.FriendlyName)).ToList();
				}

				string cleanBranchName(string name)
				{
					if (name.StartsWith(GitSettings.OriginName, StringComparison.InvariantCultureIgnoreCase)) 
						name = name.Substring(7);
					return name.Trim('/');
				}

			}
			catch (Exception e)
			{
				LogEvent(false, "Enumerate branches", "Failed to enumerate all remote branches from git repository", e);
				return new List<string>();
			}
		}



		/// <summary>
		/// Pull the branch to get all the latest changes.
		/// Exceptions all handled!
		/// </summary>
		/// <param name="directory"></param>
		/// <param name="config"></param>
		private void PullDirectory(string directory)
		{
			try
			{
				using (var repo = new Repository(directory))
				{
					var result = LibGit2Sharp.Commands.Pull(
						repo,
						Signature,
						new PullOptions()
						{
							MergeOptions = new MergeOptions()
							{
								FastForwardStrategy = FastForwardStrategy.Default
							},
							FetchOptions = new FetchOptions()
							{
								CredentialsProvider = CredentialsHandler
							}
						}
					);

					if (result.Status != MergeStatus.UpToDate)
					{
						LogEvent(true, "Pull updates", $"New content pulled, commit '{result.Commit?.MessageShort}', branch '{Branch}'");
					}
				}
			}
			catch (Exception e)
			{
				LogEvent(false, "Pull updates", $"Failed to pull updates from git repository, target branch '{Branch}'", e);
			}
		}




		/// <summary>
		/// Clone the repository from scratch.
		/// </summary>
		/// <param name="directory"></param>
		/// <param name="config"></param>
		private void CloneDirectory(string directory)
		{
			Repository.Clone(Settings.Url, directory, new CloneOptions()
			{
				BranchName = Branch,
				CredentialsProvider = this.CredentialsHandler
			});
		}




		/// <summary>
		/// Saves an event for a file to the file log and the global log
		/// </summary>
		/// <param name="filename"></param>
		/// <param name="type"></param>
		/// <param name="details"></param>
		/// <param name="e"></param>
		private void LogEvent(bool success, string title, string description, Exception e = null)
		{
			// Return event to beebox
			if (e == null)
			{
				LogItems.Add(new LogItem { Title = title, Description = description, IsError = !success });
			}
			else
			{
				LogItems.Add(new LogItem { Title = title, Description = $"{description} - {e.GetBaseException().Message}", IsError = true });
			}

			try
			{
				// Exception details
				string error = null;
				if (e != null)
				{
					StringBuilder sb = new StringBuilder();
					GetExceptionDetails(e, sb);
					error = sb.ToString();
				}

				string msg = string.Format("{0}|{1}|{2}|{3}|{4}\r\n",
						DateTime.Now.ToString(),
						success ? "SUCCESS": "FAILURE",
						title,
						description,
						error
				);

				string path = Path.Combine(LogDirectory, $"git-connector-{DateTime.Now:yyyy-MM}.log");

				// Write to global log
				File.AppendAllText(path, msg);
			}
			catch { /* Unable to write log file... */ }
		}



		/// <summary>
		/// Gets full exception details including all inner exceptions
		/// </summary>
		/// <param name="e"></param>
		/// <param name="sb">output written here</param>
		private void GetExceptionDetails(Exception e, StringBuilder sb)
		{
			sb.AppendLine($"Exception: {e.Message}");
			sb.AppendLine($"Source: {e.Source}");
			sb.AppendLine($"StackTrace: {e.StackTrace}");
			sb.AppendLine();
			if (e.InnerException != null) GetExceptionDetails(e.InnerException, sb);
		}


	}
}
