﻿using Newtonsoft.Json;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

namespace Wordbee.Beebox.Extensions.GitPackage
{
	/// <summary>
	/// Configuration of git connector in a project
	/// </summary>
	public partial class GitSettings
	{
		/// <summary>
		/// Configuration of git connector in a project
		/// </summary>
		public class MultibranchSettings
		{
			[JsonIgnore]
			private List<Regex> _multibranchRegex;


			/// <summary>
			/// The GIT url
			/// </summary>
			[JsonProperty(PropertyName = "enabled")]
			public bool Enabled { get; set; }


			/// <summary>
			/// The username used for GIT authentication
			/// </summary>
			[JsonProperty(PropertyName = "staticBranches")]
			public List<string> StaticBranchNames { get; set; }


			/// <summary>
			/// The username used for GIT authentication
			/// </summary>
			[JsonProperty(PropertyName = "regexBranches")]
			public List<string> RegexBranchNames { get; set; }



			/// <summary>
			/// Dynamically loaded regex expressions
			/// </summary>
			[JsonIgnore]
			public List<Regex> RegexBranchInstances
			{
				get
				{
					if (_multibranchRegex == null)
						PrepareRegex();
					return _multibranchRegex;
				}
			}



			/// <summary>
			/// Check if the branch name matches a statis or regex pattern
			/// </summary>
			/// <param name="branch"></param>
			/// <returns></returns>
			public bool IsMatch(string branch)
			{
				if (!Enabled) return false;
				if (StaticBranchNames?.Contains(branch) == true)
					return true;
				if (RegexBranchInstances.Any(r => r.IsMatch(branch)))
					return true;
				return false;
			}


			/// <summary>
			/// Get all branches that match a static or regex pattern.
			/// </summary>
			/// <param name="branches"></param>
			/// <returns></returns>
			public IEnumerable<string> FilterMatchingBranches(IEnumerable<string> branches)
				=> branches.Where(b => IsMatch(b));




			/// <summary>
			/// Validate parameters
			/// </summary>
			/// <returns></returns>
			public string Validate()
			{
				StaticBranchNames = TrimList(StaticBranchNames);
				RegexBranchNames = TrimList(RegexBranchNames);

				string error = PrepareRegex();
				if (error != null)
					return error;

				return null;
			}



			/// <summary>
			/// Create regex instances for regex patterns.
			/// </summary>
			/// <returns>Error is any pattern is invalid</returns>
			private string PrepareRegex()
			{
				_multibranchRegex = new List<Regex>();
				foreach (var pattern in RegexBranchNames)
				{
					try 
					{ 
						var regex = new Regex(pattern);
						_multibranchRegex.Add(regex);
					} 
					catch { return $"Invalid branch regex pattern: {pattern}"; }
				}
				return null;
			}


			/// <summary>
			/// Trim list and make unique
			/// </summary>
			/// <param name="list"></param>
			/// <returns></returns>
			private List<string> TrimList(List<string> list)
			{
				if (list != null)
					list = list.Where(b => !string.IsNullOrWhiteSpace(b)).Select(b => b.Trim()).Distinct().ToList();
				if (list == null || !list.Any())
					list = null;
				return list;
			}


		}
	}
}


