﻿using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using Wordbee.Beebox.Extensibility;

namespace Wordbee.Beebox.Extensions.GitPackage
{
	/// <summary>
	/// Implements an auto-run extension that syncs the Beebox project with a GIT repository and a specific branch.
	/// </summary>
	public partial class GitFileProcessor : BeeboxAutoRunAction
	{

		/// <summary>
		/// Gets the globally unique id of this extension.
		/// </summary>
		public override string ExtensionID { get { return "wb-git-connector"; } }


		/// <summary>
		/// Gets if the extension is by default enabled. If false, then it must be enabled explicitly in each
		/// Beebox project from the Beebox Admin UI.
		/// </summary>
		public override bool EnableByDefault { get { return false; } }



		/// <summary>
		/// Gets a name. Displayed in the Beebox extension manager page.
		/// </summary>
		public override string Name { get { return "GIT Connector"; } }



		/// <summary>
		/// Gets a description. Displayed in the Beebox extension manager page.
		/// </summary>
		public override string Description { get { return "Connects a Beebox project to a GIT repository and one or more branches."; } }


		/// <summary>
		/// Gets version number. Displayed in the Beebox extension manager page.
		/// </summary>
		public override string Version { get { return "2.00"; } }


		/// <summary>
		/// Gets the author. Displayed in the Beebox extension manager page.
		/// </summary>
		public override string Author { get { return "Wordbee"; } }



		/// <summary>
		/// Gets optional help description (not html!) of the configuration parameter.
		/// By returning a non-null value, you explicitly tell the Beebox that this extension has a configurable parameter.
		/// </summary>
		public override string ParameterHelp => 
				"Specify json object with these properties:\n" +
				"\"url\": the git url,\n \"username\": git user name,\n \"password\": git password,\n \"branch\": git branch name\n" +
				"\"multibranch\": see online documentation\n";



		/// <summary>
		/// Gets the default value of the parameter. Default can be changed by user.
		/// </summary>
		public override string ParameterDefault => "{}";



		/// <summary>
		/// Validates the user parameters.
		/// Return null if ok, otherwise return an error text.
		/// </summary>
		/// <param name="parameter">The parameter edited by a user.</param>
		/// <returns>Null: parameter is ok. Otherwise return an error message.</returns>
		public override string ValidateParameter(string parameter)
		{
			try
			{
				var settings = ToSettings(parameter);
				return settings.Validate();
			}
			catch (Exception e)
			{
				return $"Parameters seem to be malformatted: {e.Message}";
			}
		}





		/// <summary>
		/// Override to implement a daemon operation that will be executed in regular intervals.
		/// Optionally return information/errors as a list. These will be logged in the Beebox "Recent events" page. Errors will
		/// be written to the Windows event log.
		/// 
		/// You can also raise exceptions. These will be logged as well.
		/// </summary>
		/// <param name="inDirectory"></param>
		/// <param name="outDirectory"></param>
		/// <param name="logDirectory"></param>
		/// <param name="initial"></param>
		/// <param name="parameters"></param>
		/// <returns>
		/// Null or a list of events to be logged in Beebox. Each element:
		/// - Item 1: true = success, false = failure
		/// - Item 2: Mandatory. Event message (keep it short)
		/// - Item 3: Optional. Null or an additional event information.
		/// </returns>
		public override List<LogItem> Run(string inDirectory, string outDirectory, string logDirectory, bool initial, IExtensionConfiguration parameters)
		{
			var settings = ToSettings(parameters.Parameter);

			if (settings.Multibranch.Enabled)
			{
				return new MultiBranchProcessor().Run(inDirectory, logDirectory, initial, settings);
			}
			else
			{
				return new SingleBranchProcessor().Run(inDirectory, logDirectory, initial, settings);
			}
		}




		/// <summary>
		/// Deserialise parameters to settings instance
		/// </summary>
		/// <param name="parameters"></param>
		/// <returns></returns>
		private GitSettings ToSettings(string parameters)
		{
			parameters = parameters ?? ParameterDefault;
			return JObject.Parse(parameters).ToObject<GitSettings>();
		}


	}
}




