﻿using System.Collections.Generic;
using System.IO;
using System.Linq;
using Wordbee.Beebox.Extensibility;

namespace Wordbee.Beebox.Extensions.GitPackage
{
	public class MultiBranchProcessor
	{

		/// <summary>
		/// To synchronize "in" folder with any number of branches.
		/// "in" contains subfolders which are the branch names.
		/// Branch "features/1000" maps to \in\features\1000\., which contains the \.git sub-directory
		/// </summary>
		/// <param name="inDirectory"></param>
		/// <param name="logDirectory"></param>
		/// <param name="initial"></param>
		/// <param name="settings"></param>
		/// <returns>
		/// Events for the results
		/// </returns>
		public List<LogItem> Run(string inDirectory, string logDirectory, bool initial, GitSettings settings)
		{
			var logItems = new List<LogItem>();

			// Get proxy to main branch
			var proxyMain = GetBranchProxy(inDirectory, settings.BranchName, logDirectory, settings);
			proxyMain.SyncGitToBeebox();

			// Get list of remote branches we want to translate
			var branches = settings.Multibranch.FilterMatchingBranches(proxyMain.GetRemoteBranches()).ToList();
			logItems.AddRange(proxyMain.LogItems);

			// Sync each branch
			foreach (var branch in branches)
			{
				var proxy = GetBranchProxy(inDirectory, branch, logDirectory, settings);

				if (initial)
				{
					// Before the Autorun we pull any new content
					proxy.SyncGitToBeebox();
				}
				else
				{
					// At end of Autorun we push any finished translations
					proxy.SyncBeeboxToGit();
				}

				logItems.AddRange(proxy.LogItems);
			}

			return logItems;
		}



		/// <summary>
		/// Get proxy for a specific branch:
		/// - remote: A git branch
		/// - local: Sub folder inside "in" that is the branch name (/ => converts to folder breakpoint)
		/// </summary>
		/// <param name="inDirectory"></param>
		/// <param name="branchName"></param>
		/// <param name="logDirectory"></param>
		/// <param name="settings"></param>
		/// <returns></returns>
		private GitProxy GetBranchProxy(string inDirectory, string branchName, string logDirectory, GitSettings settings)
		{
			string branchFixed = branchName.Replace("/", "\\");
			string localFolder = Path.Combine(inDirectory, branchFixed);
			CreateDirectory(inDirectory, branchFixed);
			return new GitProxy(settings, localFolder, logDirectory, branchName);
		}



		/// <summary>
		/// Create the directory within the basepath folder.
		/// </summary>
		/// <param name="basepath"></param>
		/// <param name="directory"></param>
		private void CreateDirectory(string basepath, string directory)
		{
			if (string.IsNullOrEmpty(directory)) return;
			string path = Path.Combine(basepath, directory);
			if (Directory.Exists(path)) return;
			CreateDirectory(basepath, Path.GetDirectoryName(directory));
			Directory.CreateDirectory(path);
		}


	}
}
